#include "include/MathEx.js"

// StringEx
//
// 【概要】
//   String オブジェクトを拡張する
//
// 【使用例】
// Alert(String.Format("%03d, %03d, %03d", 1, 23, 456));

// 文字列の前後空白を除去
String.prototype.trim = (function() {
	var reg = /^[ \t]+|[ \t]+$/g;
	return function() {
		return this.replace(reg, '');
	}
})();

// 大文字小文字を区別しない文字列一致判定
String.prototype.equalsIgnoreCase = function(s) {
	return this.toUpperCase() === s.toUpperCase();
};

// 開始文字列一致判定
String.prototype.startsWith = function(s, isIgnoreCase) {
	if (!isIgnoreCase) {
		return this.length >= s.length && this.substring(0, s.length) === s;
	} else {
		return this.length >= s.length && s.equalsIgnoreCase(this.substring(0, s.length));
	}
};

// 最後尾文字列一致判定
String.prototype.endsWith = function(s, isIgnoreCase) {
	if (!isIgnoreCase) {
		return this.length >= s.length && this.substring(this.length - s.length, this.length) === s;
	} else {
		return this.length >= s.length && s.equalsIgnoreCase(this.substring(this.length - s.length, this.length));
	}
};

// 出現回数カウント
String.prototype.count = function(s) {
	if (s) {
		//var reg = new RegExp(s, "g");
		//return (this.length - this.replace(reg, '').length) / s.length;
		var index = 0;
		var count = 0;
		while (index >= 0) {
			index = this.indexOf(s, index);
			if (index >= 0) {
				index += s.length;
				count++;
			}
		}
		return count;
	}
	return 0;
};

// 繰り返し文字列を取得
String.prototype.repeat = function(n) {
	var t = this;
	var s = "";
	for (var i=1; i<=n; i*=2) {
		if (i > 1) {
			t = t+t;
		}
		if (n & i) {
			s += t;
		}
	}
	return s;
};

/**
 * 表示上の幅を取得する
 * @param {Number} from      開始位置(タブ移動量計算用)．
 * @param {Number} tab2space タブ幅(初回のみ指定すればOK)．
 * @returns {Number} 表示幅(fromは含まない)
 */
String.prototype.width = (function(){
	var tab = 4;
	var reg = /[\uD800-\uDBFF][\uDC00-\uDFFF]|[^\u0000-\u0080\uF8F0\uFF61-\uFF9F\uF8F1-\uF8F3]/g;
	return function(from, tab2space){
		tab = tab2space || tab;
		from = from || 0;
		var a = this.split('\t');
		var n = a[0].replace(reg, '  ').length + from;
		for (var i=1, len=a.length; i<len; i++) {
			n += (tab - n%tab) + a[i].replace(reg, '  ').length;
		}
		return n - from;
	}
})();

/**
 * 位置寄せ．
 * @param {Number} n 割り当てる領域．
 * @param {String} [align="left"] 寄せ方．
 * <UL>
 * <LI>"left"   左寄せ．
 * <LI>"center" 中央寄せ．
 * <LI>"right"  右寄せ．
 * </UL>
 * @returns {String} 整形した文字列．
 */
String.prototype.align = function(n, align) {
	align = align || 'left';
	n = n - this.width();
	if (n > 0) {
		switch (align) {
		case 'left':
			return this + ' '.repeat(n);
		case 'right':
			return ' '.repeat(n) + this;
		case 'center':
			return ' '.repeat(n/2) + this + ' '.repeat((n+1)/2);
		}
	}
	return this;
};

// いんちきなフォーマット文
String.Format = function(fmt) {
	// 0 で埋めたり寄せたり
	var _pad = function(sign, num, digit, zero, right) {
		var pad = digit - (sign+num).length;
		if (zero) {
			return sign + '0'.repeat(pad) + num;
		} else {
			return right ? sign + num + ' '.repeat(pad) : ' '.repeat(pad) + sign + num;
		}
	};
	var args = arguments;
	var count = 0;
	return fmt.replace(/%%|%([\-\+]*)(0?)(\d*)d|%([\-\+]*)(0?)(\d*)(\.?\d*)f|%(-?)(\d*)s/g, function(all, d1, d2, d3, f1, f2, f3, f4, s1, s2) {
		var type = all.slice(-1);
		switch (type) {
		case '%':
			return '%';
		case 'd':
			var digit = Number(d3 || 0);
			var arg = Number(args[++count]);
			var text = ((d1.count('+') && arg>=0) ? '+' : '') + String(arg);
			if (text.length < digit) {
				if (d2) {
					var sign = /^[\+\-]/.test(text) ? 1 : 0;
					text = text.substring(0, sign) + '0'.repeat(digit-text.length-sign) + text.substring(sign);
				} else {
					text = text.align(digit, a.count('-') ? 'left' : 'right');
				}
			}
			return text;
		case 'f':
			var arg = args[++count];
			var sign = arg < 0 ? '-' : (f1.count('+') ? '+' : '');
			var right = f1.count('-') >= 0;
			var zero = f2.length != 0;
			var num = Math.abs(arg);
			var int = Math.floor(arg);						// 整数部
			var deg = arg - int;							// 小数部
			var totalDigit = f3 ? Number(f3) : -1;			// 全体桁数
			var degDigit = f4 ? Number(f4.substring(1)) : -1;	// 小数部桁数
			if (totalDigit < 0 && degDigit < 0) {			// 桁数指定なし
				return sign + String(num);
			} else if (degDigit < 0) {						// 全体桁数だけ指定
				var text = String(num);
				if ((sign + text).length > totalDigit) {	// 桁数内に収まらない分カット
					return (sign + text).substring(0, totalDigit);
				}
				return _pad(sign, num, totalDigit, zero, right);
			} else if (totalDigit < 0) {					// 小数部だけ指定
				return sign + String(int) + String(Math.round(deg, -degDigit)).substr(1, degDigit+1);
			} else {										// 両方指定
				var text = _pad(sign, String(int)+String(Math.round(deg, -degDigit)).substring(1), totalDigit, zero, right);
				return text.length > totalDigit ? text.substring(0, totalDigit) : text;
			}
			return String(args[++count]);
		case 's':
			return _pad('', String(args[++count]), Number(s2), false, s1);
		}
		Alert('対応していないフォーマットが指定されました。\n' + all);
		Quit();
	});
}

// String のサロゲート対応
// 内部コード位置⇒文字単位位置
String.prototype.ptCodeToChar = (function() {
	var reg = /[\uD800-\uDBFF][\uDC00-\uDFFF]/g;
	return function(n) {
		return this.substring(0, n).replace(reg, ' ').length;
	}
})();
// 文字単位位置⇒内部コード位置
String.prototype.ptCharToCode = (function(){
	var reg1 = / /g;
	var reg2 = /[\uD800-\uDBFF][\uDC00-\uDFFF]/g;
	return function(n) {
		var t = ((n+1)*2 < this.length) ? this.substring(0, (n+1)*2) : this;
		t = t.replace(reg1, '-').replace(reg2, ' ').substring(0, n);

		return n*2 - t.replace(reg1, '').length;
	}
})();
// 文字単位位置⇒内部コード位置 (末尾から逆位置)
String.prototype.ptCharToCode_r = (function(){
	var reg1 = / /g;
	var reg2 = /[\uD800-\uDBFF][\uDC00-\uDFFF]/g;
	return function(n) {
		var t = ((n+1)*2 < this.length) ? this.slice(-(n+1)*2) : this;
		t = t.replace(reg1, '-').replace(reg2, ' ').slice(-n);

		return this.length - (n*2 - t.replace(reg1, '').length);
	}
})();
// 文字コードの配列を取得
String.prototype.toCodeArray = function() {
	var a = [];
	for (var i=0, len=this.length; i<len; i++) {
		a[i] = this.charCodeAt(i);
	}
	return a;
};
// 文字コード配列から文字列生成
String.fromCodeArray = (function() {
	var func = new Function("a", "to", "res",
		"for(var i=0,len=0;i<to;){" + 
		"res[len++]=String.fromCharCode(a[i++]"+(",a[i++]".repeat(64))+");" +
		"}");
	return function(a) {
		var len = a.length;
		var len2 = len - len % 65;
		var res = [];
		func(a, len2, res);
		var s = "";
		for (var i=len2; i<len; i++) {
			s += String.fromCharCode(a[i]);
		}
		return res.join("") + s;
	}
})();
// コードポイント配列の取得
String.prototype.toCodeArray_s = function() {
	var a = [];
	var alen = 0;
	for (var i=0, len=this.length; i<len;) {
		var c = this.charCodeAt(i++);
		if (0xD800 <= c && c <= 0xDBFF) {
			a[alen++] = (((c&0x3FF)<<10) | (this.charCodeAt(i++)&0x3FF)) + 0x10000;
		} else {
			a[alen++] = c;
		}
	}
	return a;
};
// コードポイント配列から文字列生成
String.fromCodeArray_s = function(a) {
	// Char 単位に分解
	var aa = [];
	var b = 0;
	for (var i=0, j=0, len=a.length; i<len; i++) {
		// サロゲートペアがほとんどないという前提で最適化
		// サロゲートペアの混入率が 1/800 を超えると逆に遅くなる
		if (a[i] > 0x10000) {
			if (i-b > 1) {
				aa.concat(a.slice(b, i));
			}
			var c = a[i];
			aa[aa.length] = ((c-0x10000)>>10) | 0xD800;
			aa[aa.length] = (c & 0x3FF) | 0xDC00;
			b = i;
		}
	}
	if (b == 0) {
		aa = a;
	} else if (len-b > 1) {
		aa.concat(a.slice(b));
	}

	// サロゲートペアの混入率が高いときは最適化していないコードの方が早い
	//for (var i=0, j=0, len=a.length; i<len; i++) {
	//	var c = a[i];
	//	if (c < 0x10000) {
	//		aa[j++] = c;
	//	} else {
	//		aa[j++] = ((c-0x10000)>>10) | 0xD800;
	//		aa[j++] = (c & 0x3FF) | 0xDC00;
	//	}
	//}
	return String.fromCodeArray(aa);
};
// サロゲート対応 length
String.prototype.length_s = (function() {
	var reg = /[\uD800-\uDBFF][\uDC00-\uDFFF]/g;
	return function() {
		return this.replace(reg, ' ').length;
	}
})();
// サロゲート対応 substring
String.prototype.substring_s = function(from, to) {
	from = from > 0 ? this.ptCharToCode(from) : 0;
	to = to > 0 ? this.ptCharToCode(to) : to;
	return this.substring(from, to);
};
// サロゲート対応 slice
String.prototype.slice_s = function(from, to) {
	from = from > 0 ? this.ptCharToCode(from) : from < 0 ? this.ptCharToCode_r(-from) : 0;
	to = to > 0 ? this.ptCharToCode(to) : to < 0 ? this.ptCharToCode_r(-to) : to;
	return this.slice(from, to);
};
// サロゲート対応 substr
// ※from の負数に対応 (本来 JScript では未対応)
String.prototype.substr_s = function(from, length) {
	var to = length==null ? undefined : (from < 0 ? (from+length>=0 ? undefined : from+length) : from+length);
	to = to < 0 ? this.ptCharToCode_r(-to) : to;
	from = from > 0 ? this.ptCharToCode(from) : from < 0 ? this.ptCharToCode_r(-from) : 0;

	return this.substring(from, to);
};
// サロゲート対応 indexOf
String.prototype.indexOf_s = function(key, from) {
	var index = this.indexOf(key, from ? this.ptCharToCode(from) : 0);
	if (index <= 0) {
		return index;
	}
	return this.ptCodeToChar(index);
};
// サロゲート対応 lastIndexOf
String.prototype.lastIndexOf_s = function(key, from) {
	var index = this.lastIndexOf(key, from ? this.ptCharToCode(from) : 0);
	if (index <= 0) {
		return index;
	}
	return this.ptCodeToChar(index);
};
// サロゲート対応 search
String.prototype.search_s = function(reg) {
	var index = this.search(reg);
	if (index <= 0) {
		return index;
	}
	return this.ptCodeToChar(index);
};
// サロゲート対応 charAt
String.prototype.charAt_s = function(n) {
	n = this.ptCharToCode(n);
	var c = this.charCodeAt(n);
	if (0xD800 <= c && c <= 0xDBFF) {
		return this.substring(n, n+2);
	}
	return this.charAt(n);
};
// サロゲート対応 charCodeAt
String.prototype.charCodeAt_s = function(n) {
	n = this.ptCharToCode(n);
	var c = this.charCodeAt(n);
	if (0xD800 <= c && c <= 0xDBFF) {
		return (((c&0x3FF)<<10) | (this.charCodeAt(n+1)&0x3FF)) + 0x10000;
	}
	return c;
};
// サロゲート対応 fromCharCode
String.fromCharCode_s = function() {
	var s = '';
	for (var i=0, len=arguments.length; i<len; i++) {
		var c = arguments[i];
		if (c < 0x10000) {
			s += String.fromCharCode(c);
		} else {
			s += String.fromCharCode(((c-0x10000)>>10) | 0xD800, (c & 0x3FF) | 0xDC00);
		}
	}
	return s;
};
// サロゲート対応 split
String.prototype.split_s = (function() {
	var reg = /([\uD800-\uDBFF][\uDC00-\uDFFF]|.)/g;
	return function(sep, limit) {
		// 文字単位での分割の場合のみサロゲート処理
		if (sep === '') {
			var a = this.match(reg) || [];
			if (limit && a.length > limit) {
				return a.slice(0, limit);
			}
			return a;
		}
		return this.split(sep, limit);
	}
})();

// 文字列を UTF-16(BigEndian, BOMなし) として CRC32 を算出する
String.prototype.GetCRC = (function() {
	var table =[0x00000000,0x77073096,0xEE0E612C,0x990951BA,0x076DC419,0x706AF48F,0xE963A535,0x9E6495A3,0x0EDB8832,0x79DCB8A4,0xE0D5E91E,0x97D2D988,0x09B64C2B,0x7EB17CBD,0xE7B82D07,0x90BF1D91,
				0x1DB71064,0x6AB020F2,0xF3B97148,0x84BE41DE,0x1ADAD47D,0x6DDDE4EB,0xF4D4B551,0x83D385C7,0x136C9856,0x646BA8C0,0xFD62F97A,0x8A65C9EC,0x14015C4F,0x63066CD9,0xFA0F3D63,0x8D080DF5,
				0x3B6E20C8,0x4C69105E,0xD56041E4,0xA2677172,0x3C03E4D1,0x4B04D447,0xD20D85FD,0xA50AB56B,0x35B5A8FA,0x42B2986C,0xDBBBC9D6,0xACBCF940,0x32D86CE3,0x45DF5C75,0xDCD60DCF,0xABD13D59,
				0x26D930AC,0x51DE003A,0xC8D75180,0xBFD06116,0x21B4F4B5,0x56B3C423,0xCFBA9599,0xB8BDA50F,0x2802B89E,0x5F058808,0xC60CD9B2,0xB10BE924,0x2F6F7C87,0x58684C11,0xC1611DAB,0xB6662D3D,
				0x76DC4190,0x01DB7106,0x98D220BC,0xEFD5102A,0x71B18589,0x06B6B51F,0x9FBFE4A5,0xE8B8D433,0x7807C9A2,0x0F00F934,0x9609A88E,0xE10E9818,0x7F6A0DBB,0x086D3D2D,0x91646C97,0xE6635C01,
				0x6B6B51F4,0x1C6C6162,0x856530D8,0xF262004E,0x6C0695ED,0x1B01A57B,0x8208F4C1,0xF50FC457,0x65B0D9C6,0x12B7E950,0x8BBEB8EA,0xFCB9887C,0x62DD1DDF,0x15DA2D49,0x8CD37CF3,0xFBD44C65,
				0x4DB26158,0x3AB551CE,0xA3BC0074,0xD4BB30E2,0x4ADFA541,0x3DD895D7,0xA4D1C46D,0xD3D6F4FB,0x4369E96A,0x346ED9FC,0xAD678846,0xDA60B8D0,0x44042D73,0x33031DE5,0xAA0A4C5F,0xDD0D7CC9,
				0x5005713C,0x270241AA,0xBE0B1010,0xC90C2086,0x5768B525,0x206F85B3,0xB966D409,0xCE61E49F,0x5EDEF90E,0x29D9C998,0xB0D09822,0xC7D7A8B4,0x59B33D17,0x2EB40D81,0xB7BD5C3B,0xC0BA6CAD,
				0xEDB88320,0x9ABFB3B6,0x03B6E20C,0x74B1D29A,0xEAD54739,0x9DD277AF,0x04DB2615,0x73DC1683,0xE3630B12,0x94643B84,0x0D6D6A3E,0x7A6A5AA8,0xE40ECF0B,0x9309FF9D,0x0A00AE27,0x7D079EB1,
				0xF00F9344,0x8708A3D2,0x1E01F268,0x6906C2FE,0xF762575D,0x806567CB,0x196C3671,0x6E6B06E7,0xFED41B76,0x89D32BE0,0x10DA7A5A,0x67DD4ACC,0xF9B9DF6F,0x8EBEEFF9,0x17B7BE43,0x60B08ED5,
				0xD6D6A3E8,0xA1D1937E,0x38D8C2C4,0x4FDFF252,0xD1BB67F1,0xA6BC5767,0x3FB506DD,0x48B2364B,0xD80D2BDA,0xAF0A1B4C,0x36034AF6,0x41047A60,0xDF60EFC3,0xA867DF55,0x316E8EEF,0x4669BE79,
				0xCB61B38C,0xBC66831A,0x256FD2A0,0x5268E236,0xCC0C7795,0xBB0B4703,0x220216B9,0x5505262F,0xC5BA3BBE,0xB2BD0B28,0x2BB45A92,0x5CB36A04,0xC2D7FFA7,0xB5D0CF31,0x2CD99E8B,0x5BDEAE1D,
				0x9B64C2B0,0xEC63F226,0x756AA39C,0x026D930A,0x9C0906A9,0xEB0E363F,0x72076785,0x05005713,0x95BF4A82,0xE2B87A14,0x7BB12BAE,0x0CB61B38,0x92D28E9B,0xE5D5BE0D,0x7CDCEFB7,0x0BDBDF21,
				0x86D3D2D4,0xF1D4E242,0x68DDB3F8,0x1FDA836E,0x81BE16CD,0xF6B9265B,0x6FB077E1,0x18B74777,0x88085AE6,0xFF0F6A70,0x66063BCA,0x11010B5C,0x8F659EFF,0xF862AE69,0x616BFFD3,0x166CCF45,
				0xA00AE278,0xD70DD2EE,0x4E048354,0x3903B3C2,0xA7672661,0xD06016F7,0x4969474D,0x3E6E77DB,0xAED16A4A,0xD9D65ADC,0x40DF0B66,0x37D83BF0,0xA9BCAE53,0xDEBB9EC5,0x47B2CF7F,0x30B5FFE9,
				0xBDBDF21C,0xCABAC28A,0x53B39330,0x24B4A3A6,0xBAD03605,0xCDD70693,0x54DE5729,0x23D967BF,0xB3667A2E,0xC4614AB8,0x5D681B02,0x2A6F2B94,0xB40BBE37,0xC30C8EA1,0x5A05DF1B,0x2D02EF8D];

	return function(isHex) {
		var crc = -1;
		for (var i=0, len=this.length; i<len; i++) {
			var c = this.charCodeAt(i);
			crc = (crc >>> 8) ^ table[(crc ^ (c>>8)) & 0xFF];
			crc = (crc >>> 8) ^ table[(crc ^ (c&0xFF)) & 0xFF];
		}
		if (!isHex) {
			return ~crc >>> 0;
		} else {
			return ("0000000" + (~crc >>> 0).toString(16).toUpperCase()).slice(-8);
		}
	};
})();